 /**
* \file: IFeatureDiscovery.h
*
* \version: 0.1
*
* \release: $Name:$
*
* Includes the interfaces of class IFeatureDiscoveryCb providing all
* callbacks which are called by the Feature Discovery
* and which must be implemented by MC Application.
*
* \component: Unified SPI
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_IFEATURE_DISCOVERY_H_
#define USPI_IFEATURE_DISCOVERY_H_


#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>

#include <memory>

#include "FeatureDiscoveryTypes.h"


namespace adit { namespace uspi {


/**
 * This class represents an interface that every implementation
 * must subclass if they wish to implement FeatureDiscovery.
 * Blocking an API interface must be avoided because it blocks
 * further notifications.
 */
class IFeatureDiscoveryCb
{
public:
    virtual ~IFeatureDiscoveryCb() {}

    /**
     * \brief Notification that a device connection was detected.
     *
     * \param inDevice  The device object of the detected device.
     *        Use DiscoveredDevice::getEventMask() to retrieve the device type mask
     *        corresponding to the derived class
     *
     * \return Application can indicate if there was an error.
     */
    virtual DiscoveryError foundCb(std::shared_ptr<DiscoveredDevice> inDevice) = 0;

    /**
     * \brief Notification that a device has been removed.
     *
     * \param inDevice  The device object of the device lost.
     *        Use DiscoveredDevice::getEventMask() to retrieve the device type mask
     *        corresponding to the derived class
     *
     * \return Application can indicate if there was an error.
     */
    virtual DiscoveryError lostCb(std::shared_ptr<DiscoveredDevice> inDevice) = 0;

    /**
     * \brief Notification that the device found was switched to SPI feature mode
     * \note Works only for AOAP mode switch
     *
     * \param inDevice  The device object of the detected device.
     *        Use DiscoveredDevice::getEventMask() to retrieve the device type mask
     *        corresponding to the derived class
     *
     * \return Application can indicate if there was an error.
     */
    virtual DiscoveryError switchedCb(std::shared_ptr<DiscoveredDevice> inDevice) = 0;

    /**
     * \brief Notification that a device connection has changed.
     *        Could appear in case of under voltage.
     *
     * \param inDevice  The device object of the detected device.
     *        Use DiscoveredDevice::getEventMask() to retrieve the device type mask
     *        corresponding to the derived class
     *
     * \return Application can indicate if there was an error.
     */
    virtual DiscoveryError changedCb(std::shared_ptr<DiscoveredDevice> inDevice) = 0;

    /**
     * \brief Called when an error happens in the FeatureDiscovery.
     *        The callback is introduced to inform the upper layer about error cases,
     *        for which another notification, such as return values, is not possible.
    *
     * \param inErrorCode  Error code
     *
     * \return Application can indicate if there was an error.
     */
    virtual DiscoveryError errorCb(DiscoveryError inErrorCode) = 0;

private:

};


} } /* namespace adit { namespace uspi { */


#endif /* USPI_IFEATURE_DISCOVERY_H_ */
